// sketch_repetition.js – "Harmonic Oscillations"
// Theme: Repetition using a grid of particles with motion
// This sketch shows repeated particle motion with interactivity and dynamic visual connections

let particles = [];
const GRID_SIZE = 15;

function setup() {
  createCanvas(900, 900);
  frameRate(30);
  colorMode(HSB, 360, 100, 100, 255); // Enable HSB coloring for prettier gradients

  // Create a grid of particles spaced evenly across the canvas
  for (let row = 0; row < GRID_SIZE; row++) {
    for (let col = 0; col < GRID_SIZE; col++) {
      let x = map(row, 0, GRID_SIZE - 1, 50, width - 50);
      let y = map(col, 0, GRID_SIZE - 1, 50, height - 50);
      particles.push(new Particle(x, y));
    }
  }
}

function draw() {
  background(240, 240, 250); // Light pastel background

  // Update and draw each particle
  for (let p of particles) {
    p.update();
    p.display();
  }

  // Draw lines connecting nearby particles
  drawConnections();
}

// Particle class handles motion and drawing
class Particle {
  constructor(x, y) {
    this.position = createVector(x, y);       // Current position
    this.origin = createVector(x, y);         // Original grid position
    this.velocity = createVector();           // Movement velocity
    this.acceleration = createVector();       // Acceleration for physics
    this.size = random(8, 15);                // Random size per particle
    this.hue = random(180, 280);              // Unique color hue
  }

  // Update particle physics and behavior
  update() {
    // Pull particle back toward original position like a spring
    let springForce = p5.Vector.sub(this.origin, this.position).mult(0.05);
    this.acceleration.add(springForce);

    // Mouse repels particle if pressed and close
    let d = dist(mouseX, mouseY, this.position.x, this.position.y);
    if (d < 100 && mouseIsPressed) {
      let repel = p5.Vector.sub(this.position, createVector(mouseX, mouseY));
      repel.setMag(50 / d); // Stronger repulsion when closer
      this.acceleration.add(repel);
    }

    // Basic physics update with damping
    this.velocity.add(this.acceleration);
    this.velocity.mult(0.95); // Damping
    this.position.add(this.velocity);
    this.acceleration.mult(0); // Reset after update
  }

  // Draw particle as a colored circle
  display() {
    noStroke();
    fill(this.hue, 80, 70, 200); // Vibrant color with transparency
    ellipse(this.position.x, this.position.y, this.size);
  }
}

// Connect nearby particles with subtle lines
function drawConnections() {
  for (let i = 0; i < particles.length; i++) {
    for (let j = i + 1; j < particles.length; j++) {
      let a = particles[i].position;
      let b = particles[j].position;
      let distance = dist(a.x, a.y, b.x, b.y);

      if (distance < 80) {
        stroke(0, 50); // Soft gray line
        strokeWeight(map(distance, 0, 80, 2, 0.1)); // Thinner if farther
        line(a.x, a.y, b.x, b.y);
      }
    }
  }
}
