// sketch_contrast.js – "Binary Resonance"
// Theme: Contrast using spatial decisions, luminance, and pixel-based thresholding

function setup() {
  createCanvas(900, 900);
  noLoop(); // This artwork is static

  background(240); // Soft gray background
  let points = [];

  // Generate 500 random points across the canvas
  for (let i = 0; i < 500; i++) {
    let px = random(width);
    let py = random(height);
    points.push(createVector(px, py));
  }

  // Sort points to give a spiral-like visual distribution
  // (Sorting by multiplying x and y creates a radial structure)
  points.sort((a, b) => (a.x * a.y) - (b.x * b.y));

  // Draw each point with contrast logic based on its quadrant
  for (let pt of points) {
    // Contrast logic: opposing quadrants get opposing color
    let isOpposingQuadrant = (pt.x < width / 2) !== (pt.y < height / 2);
    let fillColor = isOpposingQuadrant ? color(0) : color(255, 0, 0, 150); // black or translucent red

    // Size of point increases the farther from center
    let distanceFromCenter = abs(pt.x - width / 2) + abs(pt.y - height / 2);
    let size = map(distanceFromCenter, 0, width, 2, 15);

    noStroke();
    fill(fillColor);
    ellipse(pt.x, pt.y, size);
  }

  // Draw a contrast-enhancing grid using pixel brightness
  drawContrastGrid();
}

// Enhances visual contrast with a pixel-based dotted grid overlay
function drawContrastGrid() {
  loadPixels();

  for (let y = 0; y < height; y += 5) {
    for (let x = 0; x < width; x += 5) {
      let index = (x + y * width) * 4;

      // Extract RGB from pixel array
      let r = pixels[index];
      let g = pixels[index + 1];
      let b = pixels[index + 2];

      // Compute average brightness
      let brightness = (r + g + b) / 3;

      // Draw small points with inverse contrast
      stroke(brightness > 127 ? 0 : 255); // black dots on light, white on dark
      strokeWeight(0.8);
      point(x, y);
    }
  }

  updatePixels(); // Apply all pixel changes
}
